/***************************************************************************/
/*                                                                         */
/*  idrv_fio.c                                                             */
/*                                                                         */
/*  Copyright (C) 2012 Monotype Imaging Inc. All rights reserved.          */
/*                                                                         */
/***************************************************************************/
#include <ft2build.h>
#include FT_SYSTEM_H
#include FT_CONFIG_STANDARD_LIBRARY_H
#include FT_INTERNAL_STREAM_H

#include "idrv_fio.h"

#undef DEBUG

char STREAM_PREFIX[] = "**STREAM**=";

typedef struct FILE_Stream_
{
    FILE *      fp;
    FT_Stream   stream;
} FILE_Stream;

FILE *idrv_fopen(const char *name, const char *mode)
{
    FILE_Stream *fileStream = ft_smalloc(sizeof(FILE_Stream));
    if (fileStream == NULL) 
        return NULL; 

    /* initialize to all null */
    fileStream->fp = NULL;
    fileStream->stream = NULL;

#ifdef DEBUG
    printf("IDRV_FOPEN-[%s] %p\n", name, (void *)fileStream);
#endif

    /* if name starts with stream prefix it is actually a stream pointer */
    if (!strncmp(name, STREAM_PREFIX, sizeof(STREAM_PREFIX)-1))
    {
        fileStream->stream = (FT_Stream) atol(&name[sizeof(STREAM_PREFIX)-1]);

        /* reset the stream on "open" */
        FT_Stream_Seek(fileStream->stream, 0);
    }
    else
    {
        /* do a normal open */
        fileStream->fp = ft_fopen(name, mode);
    }

    return (FILE *)fileStream;
}

int idrv_fclose(FILE *fp)
{
    int ret = 0;
    FILE_Stream *fileStream = (FILE_Stream *)fp;
    if (fileStream == NULL) 
        return EOF; 

#ifdef DEBUG
    printf("IDRV_FCLOSE %p\n", (void *)fp);
#endif

    /* do a normal close for the fp - nothing for the stream */
    if (fileStream->fp != NULL)
    {
        ret = ft_fclose(fileStream->fp);
    }
    
    /* free the structure malloc from idrv_open() */
    ft_sfree(fileStream);
    return ret;
}

long idrv_fseek(FILE *fp, unsigned long offset, int whence)
{
    FILE_Stream *fileStream = (FILE_Stream *)fp;
   
#ifdef DEBUG
    printf("IDRV_FSEEK %p - %ld\n", (void *)fp, offset);
#endif

    if (fileStream->stream != NULL)
        return FT_Stream_Seek(fileStream->stream, offset);

    if (fileStream->fp != NULL)
        return ft_fseek(fileStream->fp, offset, whence);

    /* error - no stream or fp */
    return 1;
}

unsigned long idrv_fread(unsigned char *buf, unsigned long size, unsigned long count, FILE *fp)
{
    FILE_Stream *fileStream = (FILE_Stream *)fp;

#ifdef DEBUG
    printf("IDRV_FREAD %p - %ld/%ld\n", (void *)fp, size, count);
#endif

    if (fileStream->stream != NULL)
    {
        if (FT_Err_Ok == FT_Stream_Read(fileStream->stream, buf, size*count))
            return size*count;
        else
            return 0;
    }

    if (fileStream->fp != NULL)
        return ft_fread(buf, size, count, fileStream->fp);

    /* error - no stream or fp */
    return 0;
}

void idrv_make_path(char *buff, long stream)
{
    sprintf(buff, "%s%ld", STREAM_PREFIX, stream);
}
